/* Copyright (C) 2002-2018 RealVNC Ltd.  All Rights Reserved.
*/

#ifndef __VNCTYPES_H__
#define __VNCTYPES_H__

#include "vnccommontypes.h"
#include "vnccall.h"

/**
 * \file vnctypes.h
 *
 * This file defines types used by the VNC Viewer SDK.  You should normally
 * include vncviewersdk.h instead of including this file directly.
 *
 * \see VNCPoint, VNCRectangle, VNCPixelFormat
 */

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \file keysymdef.h
 *
 * This file defines constants representing key symbols as they are transmitted
 * in the RFB protocol.  You should normally include vncviewersdk.h instead of
 * including keysymdef.h directly.
 *
 * The key symbol constants used with RFB are taken from the X Windows system.
 * You can use VNCSpecialKeyToXKeySymbol(), VNCUnicodeToXKeySymbol(),
 * VNCXKeySymbolToSpecialKey() and  VNCXKeySymbolToUnicode() to translate
 * platform-specific constants and Unicode characters to and from X key
 * symbols.
 *
 * keysymdef.h is the copyright of The Open Group.  For full copyright
 * information, please see the copyright notice in the file itself.
 *
 * \see VNCSpecialKeyToXKeySymbol, VNCUnicodeToXKeySymbol,
 * VNCXKeySymbolToSpecialKey, VNCXKeySymbolToUnicode, VNCViewerSendKeyEvent
 */
#ifndef XK_VoidSymbol
#include "keysymdef.h"
#endif

/**
 * \typedef VNCViewer
 * \brief Represents a single instance of a VNC Viewer.
 *
 * To create a VNC Viewer object, call VNCViewerCreate().
 *
 * This type is opaque; use the VNC Viewer SDK APIs to interact with it.
 *
 * \see VNCViewerCreate().
 */
typedef struct VNCViewerImplBase VNCViewer;

/**
 * \typedef VNCExtension
 * \brief Represents a registered application-defined protocol extension.
 *
 * Registering an RFB extension allows you to send and receive
 * application-defined protocol extension messages, provided that the VNC
 * Server supports the same extension.
 *
 * This type is opaque; use the VNC Viewer SDK APIs to interact with it.
 *
 * \see VNCViewerRegisterExtension()
 */
typedef struct VNCExtensionImpl VNCExtension;

/**
 * Error codes that may be notified via the VNCViewerErrorCallback().  Any
 * error indicates that the VNC session has ended and that the viewer thread is
 * about to exit.
 *
 * \see VNCViewerErrorCallback
 */
typedef enum
{
    /**
     * Placeholder value for variable initialization.
     */
    VNCViewerErrorNone = 0,

    /**
     * The server hostname could not be resolved.
     */
    VNCViewerErrorDNSFailure = 1,

    /**
     * The server network is unreachable.
     */
    VNCViewerErrorNetworkUnreachable = 2,

    /**
     * The server host is unreachable.
     */
    VNCViewerErrorHostUnreachable = 3,

    /**
     * The connection attempt timed out.
     */
    VNCViewerErrorConnectionTimedOut = 4,

    /**
     * The server host refused a TCP connection.
     */
    VNCViewerErrorConnectionRefused = 5,

    /**
     * The bearer successfully established a connection to the server, but the
     * the RFB or MirrorLink version negotiated with the VNC Server is not
     * supported.
     */
    VNCViewerErrorUnsupportedProtocolVersion = 6,

    /**
     * The credentials provided via VNCViewerAsynchronousCredentialsResult()
     * are incorrect (or you did not provide a
     * VNCViewerAsynchronousCredentialsCallback at all).
     */
    VNCViewerErrorAuthenticationFailure = 7,

    /**
     * Your VNCViewerAsynchronousVerifyServerIdentityResult() rejected the
     * server public key.
     */
    VNCViewerErrorServerIdentityVerificationFailed = 8,

    /**
     * The connection to the server was lost.
     */
    VNCViewerErrorDisconnected = 9,

    /**
     * VNCViewerReset() was called (either explicitly or from
     * VNCViewerDestroy()).
     */
    VNCViewerErrorReset = 11,

    /**
     * The server gracefully closed the RFB session.
     */
    VNCViewerErrorConnectionClosed = 12,

    /**
     * Attempted to listen on a TCP port number that was already in use.
     */
    VNCViewerErrorAddressInUse = 13,

    /**
     * The security type that has been negotiated requires an RSA key pair.
     */
    VNCViewerErrorRSAKeysRequired = 14,

    /**
     * The Viewer SDK license does not grant permission to connect to this VNC
     * server.
     */
    VNCViewerErrorNotLicensedForServer = 15,

    /**
     * You called a viewer method that may not be called before the
     * VNCViewerServerInitCallback() is notified.
     */
    VNCViewerErrorIllegalBeforeServerInit = 16,

    /**
     * A protocol error occurred when interacting with a VNC Data Relay.
     */
    VNCViewerErrorDataRelayProtocolError = 17,

    /**
     * The command string passed to VNCViewerProcessCommandString() has an
     * invalid format.
     */
    VNCViewerErrorInvalidCommandString = 18,

    /**
     * The Data Relay reported that the session ID in the channel details
     * string passed to VNCViewerProcessCommandString() is not valid
     * (probably because the channel lease has already expired).
     */
    VNCViewerErrorUnknownDataRelaySessionId = 19,

    /**
     * The Data Relay channel lease expired while waiting for the server to
     * connect to the Data Relay.
     */
    VNCViewerErrorDataRelayChannelTimeout = 20,

    /**
     * The VNC Server did not offer any of the security types that were
     * configured with VNCViewerSetSecurityTypes().
     */
    VNCViewerErrorNoMatchingSecurityTypes = 21,

    /**
     * A parameter supplied to a Viewer SDK API call is not valid.
     */
    VNCViewerErrorInvalidParameter = 23,

    /**
     * Either the mobile device skin file could not be opened, or it is not a
     * valid zip archive.
     *
     * If you encounter this error during mobile device skin development, it is
     * recommended that you enable Viewer SDK logging to determine the cause.
     *
     * \see VNCViewerSetSkin(), VNCParameterSkinDTD, VNCParameterLog,
     * VNCViewerLogCallback()
     */
    VNCViewerErrorInvalidSkinArchive = 24,

    /**
     * \brief There is an error in the skin XML in the mobile device skin.
     *
     * This may indicate that:
     *
     *  - the skin XML is not well-formed
     *  - the skin XML is not valid according to the configured DTD
     *  - the skin DTD itself is not valid
     *  - the skin XML is well-formed and valid, but not semantically correct
     *    for use as part of a mobile device skin
     *
     * If you encounter this error during mobile device skin development, it is
     * recommended that you enable Viewer SDK logging to determine the cause.
     *
     * \see VNCViewerSetSkin(), VNCParameterSkinDTD, VNCParameterLog,
     * VNCViewerLogCallback()
     */
    VNCViewerErrorInvalidSkinXml = 25,

    /**
     * An image in a mobile device skin archive could not be loaded.
     */
    VNCViewerErrorInvalidSkinImage = 26,

    /**
     * \brief General non-specific failure code.
     *
     * Where-ever possible, the Viewer SDK attempts to provide a more specific
     * error code.  If you do encounter this error, it is recommended that you
     * enable Viewer SDK logging to determine the cause.
     *
     * \see VNCParameterLog, VNCViewerLogCallback()
     */
    VNCViewerErrorFailed = 27,

    /**
     * Permission was denied by the operating system.
     */
    VNCViewerErrorPermissionDenied = 28,

    /**
     * The command string selected a bearer that could not be loaded.
     *
     * More detailed information can be found in the log.
     *
     * Often this means that the viewer SDK library was simply unable to find
     * the bearer libraries. You can explicitly provide a directory for it
     * to search, using VNCViewerSetParameter and VNCParameterBearerDirectory.
     *
     * \see VNCParameterLog, VNCViewerLogCallback(), VNCViewerSetParameter(),
     * VNCParameterBearerDirectory
     */
    VNCViewerErrorBearerLoadFailed = 29,

    /**
     * This API function is not implemented for the current viewer
     * platform.
     *
     * \see VNCParameterLog, VNCViewerLogCallback()
     */
    VNCViewerErrorNotImplemented = 30,

    /**
     * The bearer failed to establish a connection due to an authentication
     * failure.
     *
     * Note that this error indicates a transport-level failure, rather than an
     * application-level failure.  That is, the failure originates from within
     * a pluggable bearer implementation, rather than with the Viewer SDK.
     */
    VNCViewerErrorBearerAuthenticationFailed = 31,

    /**
     * The specified device is not connected via USB, or the VNC Server on
     * the device is not ready to begin a USB connection.
     */
    VNCViewerErrorUSBNotConnected = 32,

    /**
     * The bearer was unable to an load an underlying software library (e.g.
     * OEM software for driving a particular type of communications), and
     * cannot proceed without it.
     */
    VNCViewerErrorUnderlyingLibraryNotFound = 33,

    /**
     * The application called an API that may not be called while the viewer
     * thread is running.
     */
    VNCViewerErrorIllegalWhileRunning = 34,

    /**
     * The application called an API that may not be called unless the viewer
     * thread is running.
     */
    VNCViewerErrorIllegalWhileNotRunning = 35,

    /**
     * An extension message cannot be sent because the server has not enabled
     * the extension.
     */
    VNCViewerErrorExtensionNotEnabled = 36,

    /**
     * The operation is not supported by the version of RFB or MirrorLink that
     * has been negotiated for the session.
     */
    VNCViewerErrorNoProtocolSupport = 37,

    /**
     * The operation failed because the thing being added or registered already
     * exists.
     */
    VNCViewerErrorAlreadyExists = 38,

    /**
     * The operation failed because there is insufficient space available to
     * buffer the requested data.
     */
    VNCViewerErrorInsufficientBufferSpace = 39,

    /**
     * Reserved for future use.
     */
    VNCViewerErrorReserved1 = 40,

    /**
     * A license could not be added because it is not valid.
     */
    VNCViewerErrorLicenseNotValid = 41,

    /**
     * A feature of the SDK cannot be used because it is not licensed.
     */
    VNCViewerErrorFeatureNotLicensed = 42,

    /**
     * The bearer successfully established a connection to the server, but the
     * server is using some protocol other than RFB (i.e. it is not a VNC
     * Server).
     */
    VNCViewerErrorNotAnRFBServer = 43,

    /**
     * The port specified by the 'p' field of the VNC Command String does not
     * represent a valid port.
     */
    VNCViewerErrorBadPort = 44,

    /**
     * There was an error reading an RFB packet
     */
    VNCViewerErrorRFBProtocolParsingError = 45,

    /**
     * A static configuration required by the bearer has not been provided.
     */
    VNCViewerErrorBearerConfigurationNotProvided = 46,

    /**
     * A static configuration provided for the bearer is invalid.
     */
    VNCViewerErrorBearerConfigurationInvalid = 47,

    /**
     * An operation was attempted for which the MirrorLink VNC Server public
     * key is required, but VNCParameterMirrorLinkVNCServerPublicKey was not
     * set.
     *
     * The MirrorLink VNC Server public key should be obtained using MirrorLink
     * Device Attestation Protocol for the TerminalMode:VNC-Server component.
     * VNC DAP SDK may be used to perform this step.
     */
    VNCViewerErrorMirrorLinkVNCServerPublicKeyRequired = 48,

    /**
     * MirrorLink content attestation failed.
     */
    VNCViewerErrorContentAttestationFailed = 49,

    /**
     * An attempt to load the specified pluggable decoder failed.
     *
     * More detailed information can be found in the log.
     *
     * Often this means that the viewer SDK library was simply unable to find
     * the decoder library. You can explicitly provide a directory for it to
     * search, using VNCViewerSetParameter and VNCParameterDecoderDirectory.
     *
     * \see VNCParameterLog, VNCViewerLogCallback(), VNCViewerSetParameter(),
     * VNCParameterDecoderDirectory
     */
    VNCViewerErrorDecoderLoadFailed = 50,

    /**
     * An attempt was made to load a pluggable decoder for an encoding
     * already handled by a previously loaded pluggable decoder.
     *
     * It is only possible to load one pluggable decoder per encoding.
     */
    VNCViewerErrorDecoderDuplicateEncoding = 51,

    /**
     * An attempt was made to load a pluggable decoder for an unsupported
     * encoding. Only pluggable decoders which support RAW, TRLE, ZRLE,
     * JPEG, SLRLE, VAH264 or H264 are allowed.
     */
    VNCViewerErrorDecoderUnsupportedEncoding = 52,

    /**
     * A pluggable decoder indicated that it has failed. More information
     * may be available in the log.
     */
     VNCViewerErrorDecoderFailed = 53,

    /**
     * An attempt to load the specified pluggable renderer failed.
     *
     * More detailed information can be found in the log.
     *
     * Often this means that the viewer SDK library was simply unable to find
     * the renderer library. You can explicitly provide a directory for it to
     * search, using VNCViewerSetParameter and VNCParameterRendererDirectory.
     *
     * \see VNCParameterLog, VNCViewerLogCallback(), VNCViewerSetParameter(),
     * VNCParameterRendererDirectory
     */
    VNCViewerErrorRendererLoadFailed = 54,

    /**
     * An attempt was made to load a second pluggable renderer.
     *
     * It is only possible to load one pluggable renderer at a time.
     */
    VNCViewerErrorRendererAlreadyLoaded = 55,

    /**
     * A pluggable renderer indicated that it has failed. More information
     * may be available in the log.
     */
     VNCViewerErrorRendererFailed = 56,

    /**
     * It is not possible to use this feature while a previously selected one
     * is active. For example, skins and pluggable renderers may not be used
     * at the same time.
     */
     VNCViewerErrorConflictingFeature = 57,

    /**
     * It is not possible to perform this action, because there is no pluggable
     * renderer currently active.
     */
     VNCViewerErrorRendererNotActive = 58,

    /**
     * A field or value cannot be used, as it has been deprecated in the RFB or
     * MirrorLink version negotiated with the VNC Server.
     */
    VNCViewerErrorDeprecatedFieldUsed = 59,

} VNCViewerError;

/**
 * \brief Describes a mouse cursor.
 *
 * The cursor pixel data is contained in the buffer field.
 * The cursor pixel data is encoded in the following pixel format:
 * The framebuffer pixel format is used to encode the red, green and blue
 * channel of each pixel.
 * Each pixel also contains an alpha channel.
 * The alpha channel is specified by the alpha mask that was passed to
 * VNCViewerSetFrameBufferAlphaMask().
 * The framebuffer pixel format must be true color.
 * The sum of the alpha channel width (in bits) and the alpha channel shift
 * (in bits) must be less than or equal to the framebuffer pixel width (in bits)
 * The alpha channel must not overlap any of the other three channels of the
 * pixel format.
 * VNCViewerSetFrameBufferAlphaMask must be successfully called before the
 * current cursor will be passed back to the SDK client through an invocation of
 * VNCViewerCursorUpdateCallback
 *
 * \see VNCPixelFormat, VNCViewerSetFrameBufferAlphaMask,
 * VNCViewerCursorUpdateCallback
 */
typedef struct
{
  /**
   * A byte buffer containing the cursor pixel data.
   * The pixels are encoded into the buffer one row at a time,
   * starting with topmost row and ending with the bottommost row.
   * Each row is encoded starting with the leftmost pixel and ending with the
   * rightmost pixel. Each row occupies (size.width * stride) bytes.
   * The length of the buffer is therefore ((size.width * stride) * height)
   * bytes.
   */
  const vnc_uint8_t* buffer;
  /** The stride of each row of pixels in the buffer measured in bytes. */
  vnc_uint32_t stride;
  /** The size of the cursor in pixels */
  VNCPoint size;
  /**
   * The cursor hotspot.
   * The hotspot is a point lying within the bounds of the mouse cursor.
   * When a mouse click occurs, the location of the click coincides with
   * the point on the desktop directly underneath the cursor hotspot.
   * The hotspot is specified as an offset relative to the top left corner
   * of the cursor.
   * The x coordinate is measured left to right.
   */
  VNCPoint hotspot;
} VNCCursor;

/**
 * \brief Encoding numbers that may be returned by
 * VNCViewerGetProperty(VNCViewerPropertyLastUsedEncoding).
 *
 * For an explanation of these values, please see the RFB 3.8, RFB 4.1 
 * and MirrorLink (SLRLE) Specification.
 */
typedef enum
{
    /** \cond */
    VNCEncodingRaw = 0,
    VNCEncodingCopyRect = 1,
    VNCEncodingRRE = 2,
    VNCEncodingHextile = 5,
    VNCEncodingTRLE = 15,
    VNCEncodingZRLE = 16,
    VNCEncodingH264 = 20,
    VNCEncodingJPEG = 21,
    VNCEncodingJRLE = 22,
    VNCEncodingVAH264 = 0x48323634,
    VNCEncodingSLRLE = -525,
    /** \endcond */
} VNCEncoding;

/**
 * \brief Security type numbers, as supplied to VNCViewerSetSecurityTypes() and
 * returned by VNCViewerGetProperty(VNCViewerPropertySecurityType).
 *
 * During the RFB handshake, a VNC Server will offer all of the security types
 * that it supports and is configured to allow, in order of preference.  The
 * SDK selects one based on the values of the Encryption and SingleSignOn
 * parameters.
 *
 * The supported security types are:
 *
 *  - None - No authentication or encryption.
 *  - VncAuth - Legacy 8-character password authentication.  The VNC Server has
 *    a single password that is known by all its users.  Once authenticated,
 *    the session is not encrypted.
 *  - RA2 - Each user authenticates to the VNC Server with their own
 *    username and password.  Both viewer and server are
 *    required to have an RSA key pair (i.e. you must call VNCGenerateRSAKeys
 *    and then call VNCViewerSetParameter(VNCParameterRSA_Private_Key) with the
 *    result).  The RSA key pairs are used to negotiate a shared 256-bit key
 *    for the AES encryption that is used for the remainder of the session.
 *  - RA2Rev - Uses the same cryptography as RA2, optionally requires the
 *    server to be required to supply a password to the viewer (if
 *    VNCParameterAuthenticateServer is set).
 *
 * The security types ending with -ne are the 'no encryption' variants.  These
 * use the same authentication mechanisms as their encrypted counterparts, but
 * do not continue to encrypt the session after authentication is complete.
 *
 * \see VNCViewerSetSecurityTypes, VNCParameterAuthenticateServer,
 * VNCViewerPropertySecurityType
 */
typedef enum
{
    /** \cond */
    VNCSecurityTypeInvalid = 0,
    VNCSecurityTypeNone = 1,
    VNCSecurityTypeVncAuth = 2,
    VNCSecurityTypeRA2 = 5,
    VNCSecurityTypeRA2ne = 6,
    VNCSecurityTypeRA2_128 = 5,
    VNCSecurityTypeRA2ne_128 = 6,
    VNCSecurityTypeRA2Rev = 13,
    VNCSecurityTypeRA2Revne = 14,
    VNCSecurityTypeRA2_256 = 129,
    VNCSecurityTypeRA2ne_256 = 130
    /** \endcond */
} VNCSecurityType;

/**
 * \brief Properties that may be queried via VNCViewerGetProperty() or
 * VNCViewerGetPropertyString().
 *
 * Integer-valued properties must be queried with VNCViewerGetProperty().
 * Passing the identifier for an integer-valued property to
 * VNCViewerGetPropertyString() will result in a NULL return.
 *
 * String-valued properties must be queried with VNCViewerGetPropertyString().
 * Passing the identifier for a string-valued property to
 * VNCViewerGetProperty() will result in a zero return.
 */
typedef enum
{
    /**
     * \brief Whether the VNC Viewer thread is running (integer).
     *
     * The value is non-zero if the VNC Viewer has been started but has not yet
     * exited or been stopped.  Otherwise, the value is zero.
     */
    VNCViewerPropertyIsRunning,

    /**
     * \brief Whether the VNC Viewer is fully connected to a VNC Server
     * (integer).
     *
     * The value is non-zero if the VNC Viewer is connected to a VNC Server and
     * has completed the RFB handshake.  Otherwise, the value is zero.
     */
    VNCViewerPropertyIsConnected,

    /**
     * \brief The major component of the version number of the RFB protocol
     * that is actually in use for the session (integer).
     *
     * \see VNCParameterMinimumRFBVersionMajor,
     * VNCParameterMinimumRFBVersionMinor, VNCViewerPropertyRFBVersionMinor
     */
    VNCViewerPropertyRFBVersionMajor,

    /**
     * \brief The minor component of the version number of the RFB protocol
     * that is actually in use for the session (integer).
     *
     * \see VNCParameterMinimumRFBVersionMajor,
     * VNCParameterMinimumRFBVersionMinor, VNCViewerPropertyRFBVersionMajor
     */
    VNCViewerPropertyRFBVersionMinor,

    /**
     * \brief The actual unscaled width of the server desktop (integer).
     *
     * The server desktop is the area within the framebuffer into which the
     * remote display is copied.  This may be smaller than the size of the
     * entire framebuffer if a mobile device skin is in use.
     */
    VNCViewerPropertyServerDesktopWidth,

    /**
     * \brief The actual unscaled height of the server desktop (integer).
     *
     * The server desktop is the area within the framebuffer into which the
     * remote display is copied.  This may be smaller than the size of the
     * entire framebuffer if a mobile device skin is in use.
     */
    VNCViewerPropertyServerDesktopHeight,

    /**
     * \brief An approximate estimate of the connection speed, in bits per
     * second (integer).
     *
     * A returned value of 0 indicates a failure to calculate a meaningful
     * estimate.
     */
    VNCViewerPropertyLineSpeedEstimate,

    /**
     * \brief The security type in use for the session (integer).
     *
     * The returned value is a member of the VNCSecurityType enumeration.
     */
    VNCViewerPropertySecurityType,

    /**
     * \brief The RFB encoding that was last used in the session (integer).
     *
     * The returned value is a member of the VNCEncoding enumeration.
     */
    VNCViewerPropertyLastUsedEncoding,

    /**
     * \brief Non-zero if display updates are currently enabled (integer).
     *
     * \see VNCViewerEnableDisplayUpdates
     */
    VNCViewerPropertyDisplayUpdatesEnabled,

    /**
     * \brief The name of the manufacturer of the server host machine (string).
     *
     * The value of this property will be NULL up until the point in the
     * session that the server communicates the value to the viewer.  This
     * takes place after authentication but before the
     * VNCViewerServerInitCallback() is invoked.
     *
     * The value of this property will always be NULL if the server is not a
     * VNC Mobile Server.
     *
     * \see VNCViewerPropertyServerHostModel,
     * VNCViewerPropertyServerHostVersion, VNCViewerPropertyServerHostOS
     */
    VNCViewerPropertyServerHostManufacturer,

    /**
     * \brief The model name of the server host machine (string).
     *
     * The value of this property will be NULL up until the point in the
     * session that the server communicates the value to the viewer.  This
     * takes place after authentication but before the
     * VNCViewerServerInitCallback() is invoked.
     *
     * The value of this property will always be NULL if the server is not a
     * VNC Mobile Server.
     *
     * \see VNCViewerPropertyServerHostManufacturer,
     * VNCViewerPropertyServerHostVersion, VNCViewerPropertyServerHostOS
     */
    VNCViewerPropertyServerHostModel,

    /**
     * \brief The model version of the server host machine (string).
     *
     * The value of this property will be NULL up until the point in the
     * session that the server communicates the value to the viewer.  This
     * takes place after authentication but before the
     * VNCViewerServerInitCallback() is invoked.
     *
     * The value of this property will always be NULL if the server is not a
     * VNC Mobile Server.
     *
     * \see VNCViewerPropertyServerHostManufacturer,
     * VNCViewerPropertyServerHostModel, VNCViewerPropertyServerHostOS
     */
    VNCViewerPropertyServerHostVersion,

    /**
     * \brief The X-coordinate of the top-left pixel of the server desktop
     * within the framebuffer (integer).
     *
     * The server desktop is the area within the framebuffer into which the
     * remote display is copied.  Its top-left may not be at (0, 0) if a mobile
     * device skin is in use.
     */
    VNCViewerPropertyServerDesktopX,

    /**
     * \brief The Y-coordinate of the top-left pixel of the server desktop
     * within the framebuffer (integer).
     *
     * The server desktop is the area within the framebuffer into which the
     * remote display is copied.  Its top-left may not be at (0, 0) if a mobile
     * device skin is in use.
     */
    VNCViewerPropertyServerDesktopY,

    /**
     * \brief Non-zero if pointer event injection is possible for this
     * connection (integer).
     *
     * Not all connections support the injection of pointer events. For
     * example, mobile devices without touchscreens will not usually respond
     * to having pointer events injected (and if they do respond, the results
     * may not be what you expect).
     *
     * You can use this property to query whether the connection supports
     * pointer events.  Calling VNCViewerSendPointerEvent() when the connection
     * does not support pointer events will have no effect.
     *
     * \see VNCViewerSendPointerEvent()
     */
    VNCViewerPropertyServerSupportsPointerEvents,

    /**
     * \brief The public key part of the RSA key pair that was assigned to the
     * viewer (string).
     *
     * The value is a hexidecimal string.  If an RSA key pair has not been
     * assigned to the viewer, then the value is NULL.
     *
     * \see VNCParameterRSAKeys, VNCViewerPropertyViewerSignature
     */
    VNCViewerPropertyViewerPublicKey,

    /**
     * \brief The public key signature of the RSA key pair that was assigned to
     * the viewer (string).
     *
     * The signature of an RSA public key used by a VNC Viewer or VNC Server is
     * the first eight bytes of the SHA-1 hash of the public key.  If the end
     * user has physical access to both the viewer and server, they are often
     * able visually to compare the signature received by the viewer with the
     * one advertised with the server.  This gives the end user assurance that
     * the viewer is indeed connected to the correct server.
     *
     * The value is a hexidecimal string with a hyphen ('-') between each byte.
     * If an RSA key pair has not been assigned to the viewer, then the value
     * is NULL.
     *
     * \see VNCParameterRSAKeys, VNCViewerPropertyViewerSignature
     */
    VNCViewerPropertyViewerSignature,

    /**
     * \brief The RSA public key communicated to the viewer by the VNC Server
     * (string).
     *
     * The value is a hexidecimal string.  If the server has not communicated a
     * public key to the viewer (because the security type that is in use by
     * the session does not require one), then the value is NULL.
     *
     * \see VNCSecurityType, VNCViewerPropertyServerSignature
     */
    VNCViewerPropertyServerPublicKey,

    /**
     * \brief The signature of the RSA public key communicated to the viewer by
     * the VNC Server (string).
     *
     * The signature of an RSA public key used by a VNC Viewer or VNC Server is
     * the first eight bytes of the SHA-1 hash of the public key.  If the end
     * user has physical access to both the viewer and server, they are often
     * able visually to compare the signature received by the viewer with the
     * one advertised with the server.  This gives the end user assurance that
     * the viewer is indeed connected to the correct server.
     *
     * The value is a hexidecimal string with a hyphen ('-') between each byte.
     * If the server has not communicated a public key to the viewer (because
     * the security type that is in use by the session does not require one),
     * then the value is NULL.
     *
     * \see VNCSecurityType, VNCViewerPropertyServerPublicKey
     */
    VNCViewerPropertyServerSignature,

    /**
     * \brief The short name of the bearer for this session (string).
     *
     * This is the same as the value of the 't' field in the VNC Command String
     * that was passed to VNCViewerProcessCommandString().
     *
     * If the viewer thread is not running, then the value of this property is
     * NULL.
     *
     * \see VNCViewerProcessCommandString()
     */
    VNCViewerPropertyBearerName,

    /**
     * \brief A human-readable longer name for the bearer for this session
     * (string).
     *
     * If the viewer thread is not running, or if the bearer does not define a
     * value for this property, then the value of this property is NULL.
     */
    VNCViewerPropertyBearerFullName,

    /**
     * \brief A human-readable longer description for the bearer for this
     * session (string).
     *
     * If the viewer thread is not running, or if the bearer does not define a
     * value for this property, then the value of this property is NULL.
     */
    VNCViewerPropertyBearerDescription,

    /**
     * \brief A version number or other version identier for the bearer for
     * this session (string).
     *
     * If the viewer thread is not running, or if the bearer does not define a
     * value for this property, then the value of this property is NULL.
     */
    VNCViewerPropertyBearerVersion,

    /**
     * \brief A description of the network endpoint(s) on which the viewer is
     * listening for an incoming connection from a VNC Server (string).
     *
     * When either VNCViewerListen() is called, or a VNC command string with
     * t=L is passed to VNCViewerProcessCommandString(), or some other
     * pluggable bearer is in use that can be said to listen for incoming
     * connections, it may be useful for the application to know on which
     * endpoint(s) the viewer is listening, so that this information can be
     * displayed to the end user.  In these cases,
     * VNCViewerPropertyListeningInformation provides this information.
     *
     * To help determine when this property is likely to have a meaningful
     * value, register a VNCViewerSessionProgressCallback and note when it
     * notifies the VNCViewerSessionProgressListening state.
     *
     * If the bearer in use is one that does not listen for an incoming
     * connection, or the viewer is not currently running and listening, then
     * the value of this property is NULL.
     *
     * \see VNCViewerListen(), VNCViewerProcessCommandString(),
     * VNCViewerSessionProgressCallback
     */
    VNCViewerPropertyListeningInformation,

    /**
     * \brief The address of the local endpoint of the connection (string).
     *
     * The format and meaning of the returned string is dependent on the
     * bearer.  For example, with the TCP 'C' bearer, this is an IP address and
     * a port number, separated by a colon.
     *
     * If the viewer thread is not running, or if the bearer does not define a
     * value for this property, then the value of this property is NULL.
     */
    VNCViewerPropertyLocalAddress,

    /**
     * \brief The address of the remote endpoint of the connection (string).
     *
     * The format and meaning of the returned string is dependent on the
     * bearer.  For example, with the TCP 'C' bearer, this is an IP address and
     * a port number, separated by a colon.
     *
     * If the viewer thread is not running, or if the bearer does not define a
     * value for this property, then the value of this property is NULL.
     */
    VNCViewerPropertyPeerAddress,

    /**
     * \brief The current progress made by the viewer while establishing a VNC
     * session (integer).
     *
     * The returned value is a member of the VNCViewerSessionProgress
     * enumeration and is the same as that passed to the application by the
     * VNCViewerSessionProgressCallback.
     *
     * \see VNCViewerSessionProgressCallback, VNCViewerSessionProgressCallback
     */
    VNCViewerPropertySessionProgress,

    /**
     * \brief The name of the server desktop to which the viewer is connected
     * (string).
     *
     * If the viewer thread is not running, then the value of this property is
     * NULL.
     *
     * \see VNCViewerServerInitCallback
     */
    VNCViewerPropertyDesktopName,

    /**
     * \brief The operating system of the server host machine (string).
     *
     * The value of this property will be NULL up until the point in the
     * session that the server communicates the value to the viewer.  This
     * takes place after authentication but before the
     * VNCViewerServerInitCallback() is invoked.
     *
     * The value of this property will always be NULL if the server is not a
     * VNC Mobile Server.
     *
     * \see VNCViewerPropertyServerHostManufacturer,
     * VNCViewerPropertyServerHostModel, VNCViewerPropertyServerHostVersion
     */
    VNCViewerPropertyServerHostOS,

    /**
     * \brief The error code returned by the bearer (integer).
     *
     * If a session fails as the result of an error raised by a bearer, then
     * this property can be used to retrieve that error. This is useful when
     * the viewer application must have specific handling for different errors
     * that can arise at the transport level.
     *
     * If the session has not yet terminated, or was not terminated as the
     * result of a bearer error, then the value of this property is
     * VNCBearerErrorNone.
     */
    VNCViewerPropertyBearerError,

    /**
     * \brief Non-zero if touch event injection is possible for this connection
     * (integer).
     *
     * Touch events differ from pointer events in that they encapsulate
     * multiple locations (usually of fingertips) with different pressures,
     * whereas a pointer event has a single point of contact (usually of a
     * mouse pointer) with a combination of different buttons.
     *
     * Not all connections support the injection of touch events.
     * Calling VNCViewerSendTouchEvent() when the connection does not support
     * touch events will terminate the session with a
     * ::VNCViewerErrorNoProtocolSupport error.
     *
     * It is therefore recommended that, if you intend for your viewer
     * application to support touch events, you query
     * ::VNCViewerPropertyServerSupportsTouchEvents once, after the session has
     * been established, and keep a note of the result.
     *
     * \see VNCViewerSendTouchEvent()
     */
    VNCViewerPropertyServerSupportsTouchEvents,

    /**
     * \brief Non-zero if key event injection is possible for this connection
     * (integer).
     *
     * Not all connections support the injection of key events. Calling
     * VNCViewerSendKeyEvent() when the connection does not support key events
     * will terminate the session with a ::VNCViewerErrorNoProtocolSupport
     * error.
     *
     * It is therefore recommended that, if you intend for your viewer
     * application to support key events, you query
     * ::VNCViewerPropertyServerSupportsKeyEvents once, after the session has
     * been established, and keep a note of the result.
     *
     * Note that calling VNCViewerSendDeviceKeyOrKeyEvent() or
     * VNCViewerProcessWindowsKeyMessage() will not terminate the session. This
     * allows these APIs to be safely used to send device key events even when
     * the connection does not support key events.
     *
     * \see VNCViewerSendKeyEvent()
     */
    VNCViewerPropertyServerSupportsKeyEvents,

    /**
     * \brief Non-zero if device key event injection is possible for this
     * connection (integer).
     *
     * Not all connections support the injection of device key events. Calling
     * VNCViewerSendDeviceKeyEvent() when the connection does not support
     * device key events will cause a warning to be logged.
     *
     * \see VNCViewerSendDeviceKeyEvent()
     */
    VNCViewerPropertyServerSupportsDeviceKeyEvents,

    /**
     * \brief The kernel version of the server host machine (string).
     *
     * The value of this property will be NULL up until the point in the
     * session that the server communicates the value to the viewer.  This
     * takes place after authentication but before the
     * VNCViewerServerInitCallback() is invoked.
     *
     * The value of this property will always be NULL if the server is not a
     * VNC Mobile Server, or if the server operating system has no concept of
     * a kernel version.
     *
     * The format of this property is a dotted decimal string.
     *
     * \see VNCViewerPropertyServerHostManufacturer,
     * VNCViewerPropertyServerHostModel, VNCViewerPropertyServerHostVersion,
     * VNCViewerPropertyServerHostOS
     */
    VNCViewerPropertyServerHostKernelVersion,

    /**
     * \brief The serial number of the remote endpoint of the connection
     * (string).
     *
     * The format and meaning of the returned string is dependent on the
     * bearer.  For example, with the 'AAP' bearer, this is the serial number
     * of the USB device.
     *
     * If the viewer thread is not running, or if the bearer does not define a
     * value for this property, then the value of this property is NULL.
     *
     * Even when it is defined by the bearer, it may not be set until
     * VNCViewerSessionProgressNegotiatingWithVNCServer is reached.
     */
    VNCViewerPropertyDeviceSerial,

} VNCViewerProperty;

/**
 * \brief Contextual information for parameter iteration.
 *
 * This structure is used to hold state in between calls to
 * VNCViewerIterateParameterNames().  The caller allocates and zeroes a
 * VNCParameterIterationContext structure before their first call to that
 * method, and then passes the same VNCParameterIterationContext into
 * subsequent calls to continue iteration.
 *
 * \see VNCViewerIterateParameterNames()
 */
typedef struct
{
    /* \cond */
    void *p1;
    void *p2;
    /* \endcond */
} VNCParameterIterationContext;

/**
 * \brief State changes notified by VNCViewerSessionProgressCallback.
 *
 * The sequence of notifications varies according to the connection type.
 *
 * When making an outward connection to a VNC Server, the sequence is:
 *
 * -# VNCViewerSessionProgressPerformingNameLookup
 * -# VNCViewerSessionProgressConnecting
 * -# VNCViewerSessionProgressNegotiatingWithProxy
 * -# VNCViewerSessionProgressNegotiatingWithVNCServer
 *
 * When listening for an incoming connection from a VNC Server, the sequence
 * is:
 *
 * -# VNCViewerSessionProgressListening
 * -# VNCViewerSessionProgressNegotiatingWithVNCServer
 *
 * When making an outward connection via a VNC Data Relay, the sequence is:
 *
 * -# VNCViewerSessionProgressPerformingNameLookup
 * -# VNCViewerSessionProgressConnecting
 * -# VNCViewerSessionProgressNegotiatingWithProxy
 * -# VNCViewerSessionProgressNegotiatingWithDataRelay
 * -# VNCViewerSessionProgressWaitingForVNCServer
 * -# VNCViewerSessionProgressNegotiatingWithVNCServer
 *
 * In all of these cases, if one or more VNCViewerSkinFetchCallback()s has been
 * registered, there will be one further notification:
 *
 * -# VNCViewerSessionProgressFetchingSkin
 *
 * VNCViewerSessionProgressPerformingNameLookup is notified only if the peer to
 * which the Viewer SDK connects is identified by a domain name.  This is true
 * whether the SDK connects via a proxy, a VNC Data Relay, or directly to a VNC
 * Server.
 *
 * VNCViewerSessionProgressNegotiatingWithProxy is notified only if a proxy
 * server is in use for the session.
 *
 * \see VNCViewerSessionProgressCallback, VNCViewerConnect, VNCViewerListen,
 * VNCViewerProcessCommandString
 */
typedef enum
{
    /**
     * \brief Placeholder value for variable initialization.
     */
    VNCViewerSessionProgressNone = 0x0000,

    /**
     * \brief Notified when the viewer thread is about to start listening for
     * an incoming connection.
     */
    VNCViewerSessionProgressListening = 0x0100,

    /**
     * \brief Notified when the viewer thread is about to perform a name lookup
     * (e.g.  using DNS).
     */
    VNCViewerSessionProgressPerformingNameLookup = 0x0200,

    /**
     * \brief Notified when the viewer thread is about to start connecting
     * outwards.
     */
    VNCViewerSessionProgressConnecting = 0x0300,

    /**
     * \brief Notified when the viewer thread is about to start negotiating
     * with an HTTP or SOCKS proxy.
     */
    VNCViewerSessionProgressNegotiatingWithProxy = 0x0400,

    /**
     * \brief Notified when the viewer thread is about to start negotiating
     * with a VNC Data Relay.
     */
    VNCViewerSessionProgressNegotiatingWithDataRelay = 0x0500,

    /**
     * \brief Notified when the viewer thread is about to start negotiating
     * with Barry.
     *
     * Barry (http://netdirect.ca/software/packages/barry) is an open-source
     * software package for managing BlackBerry devices from Linux and other
     * UNIX machines.  It is required by the USBBB bearer on these platforms.
     *
     * \deprecated No longer used as of Mobile Solution 3.0 - BlackBerry
     * devices are no longer supported.
     */
    VNCViewerSessionProgressNegotiatingWithBarry = 0x0510,

    /**
     * \brief Notified when the viewer thread is about to start negotiating
     * with adb (the Android Debug Bridge).
     *
     * adb is one option for establishing a USB connection to an Android
     * device.
     */
    VNCViewerSessionProgressNegotiatingWithAdb = 0x0520,

    /**
     * \brief Notified when the viewer thread has finished negotiating with a
     * VNC Data Relay, and the Data Relay channel is awaiting the corresponding
     * connection from the VNC Server.
     */
    VNCViewerSessionProgressWaitingForVNCServer = 0x0600,

    /**
     * \brief Notified when the viewer thread has established a connection to
     * the VNC Server and the RFB protocol begins.
     */
    VNCViewerSessionProgressNegotiatingWithVNCServer = 0x0700,

    /**
     * \brief Notified if one or more VNCViewerSkinFetchCallback()s has been
     * registered, after the RFB protocol has completed the security
     * negotiation and it has been determined that the server is indeed a
     * mobile device.
     */
    VNCViewerSessionProgressFetchingSkin = 0x0800,

    /**
     * \brief Notified when the normal RFB handshake is complete, the skin (if
     * any) has been fetched, and the viewer is negotiating MirrorLink features
     * with the server.
     */
    VNCViewerSessionProgressMirrorLinkHandshake = 0x0900,

    /**
     * \brief Notified when the session is fully established and the skin (if
     * any) has been fetched.
     */
    VNCViewerSessionProgressSessionEstablished = 0xf000,
} VNCViewerSessionProgress;

/**
 * \brief Supplied to the VNCViewerLockRectangleExCallback and
 * VNCViewerUnlockRectangleExCallback to inform the application of the reason
 * for locking the framebuffer.
 *
 * This enumeration exists mainly to allow the application to decide when best
 * to redraw or invalidate the framebuffer.
 *
 * The RFB protocol groups updated rectangles into framebuffer updates, the
 * starts and ends of which are notified by the
 * VNCViewerFrameBufferUpdateStartCallback and the
 * VNCViewerFrameBufferUpdateEndExCallback.  For best performance, and to avoid
 * 'tearing' of the server's display in the framebuffer, locked rectangles with
 * reason VNCViewerLockRectangleReasonServer should not be redrawn or
 * invalidated until the VNCViewerFrameBufferUpdateEndExCallback.
 *
 * On the other hand, locked rectangles with reasons
 * VNCViewerLockRectangleReasonInitialization and
 * VNCViewerLockRectangleReasonSkin do not originate with the server and are
 * independent of the framebuffer update callbacks.  These rectangles should be
 * redrawn or invalidated immediately.
 *
 * Your application may safely choose to ignore this flag and redraw or
 * invalidate the server deskop immediately, at the cost of allowing tearing.
 *
 * \see VNCViewerLockRectangleExCallback, VNCViewerUnlockRectangleExCallback,
 * VNCViewerFrameBufferUpdateStartCallback,
 * VNCViewerFrameBufferUpdateEndExCallback
 */
typedef enum
{
    /**
     * \brief The rectangle has been locked so that the framebuffer can be
     * initialized.
     *
     * In this case, it is best to redraw or invalidate immediately.
     */
    VNCViewerLockRectangleReasonInitialization = 1,

    /**
     * \brief The rectangle has been locked so that a framebuffer update received
     * from the VNC Server can be being copied into the framebuffer.
     *
     * In this case, it is best to redraw or invalidate from the
     * VNCViewerFrameBufferUpdateEndExCallback, to avoid tearing.
     */
    VNCViewerLockRectangleReasonServer,

    /**
     * \brief The rectangle has been locked so that part of the device skin can
     * be being copied into the framebuffer.
     *
     * In this case, it is best to redraw or invalidate immediately.
     */
    VNCViewerLockRectangleReasonSkin,
} VNCViewerLockRectangleReason;

/**
 * \brief No invalid region tracking will be done by the SDK.
 *
 * The Viewer SDK invalid region mode of operation is set through
 * VNCViewerSetParameter, with the parameter VNCParameterInvalidRegionTracking.
 *
 * This is the default mode of operation for the Viewer SDK. No invalid
 * rectangles will be given in the frame buffer update end callback.
 *
 * In this mode, the viewer application needs to keep track of the updated
 * rectangles from the lock and unlock calls. This adds extra responsibility on
 * the client application and may result in a large number of rectangles being
 * tracked. For simplicity and performance, one of the other modes of operation
 * are prefered.
 *
 * \see VNCViewerSetParameter, VNCViewerGetParameter,
 * VNCParameterInvalidRegionTracking,
 * VNCViewerFrameBufferUpdateEndExCallback
 */
#define VNCViewerInvalidRegionTrackingNone "none"

/**
 * \brief The Viewer SDK will track a bounding rectangle which contains all the
 * rectangles that have become invalid during the frame buffer update.
 *
 * The Viewer SDK invalid region mode of operation is set through
 * VNCViewerSetParameter, with the parameter VNCParameterInvalidRegionTracking.
 *
 * In this mode of operation, the Viewer SDK will maintain a rectangle inside
 * the frame buffer which contains all the rectangles that have become invalid
 * during the frame buffer update. The SDK will return this bounding rectangle
 * through VNCViewerFrameBufferUpdateEndExCallback. The SDK can efficiently
 * monitor the updated areas and maintain the bounding rectangle with minimal
 * effort. Using this mode, as opposed to the default option, will therefore
 * lead to improved performance for the client application.
 *
 * \see VNCViewerSetParameter, VNCViewerGetParameter,
 * VNCParameterInvalidRegionTracking,
 * VNCViewerFrameBufferUpdateEndExCallback
 */
#define VNCViewerInvalidRegionTrackingBounding "bounding"

/**
 * \brief The Viewer SDK will track the region of rectangles becoming invalid
 * during an update.
 *
 * The Viewer SDK invalid region mode of operation is set through
 * VNCViewerSetParameter, with the parameter VNCParameterInvalidRegionTracking.
 *
 * In this mode of operation, the Viewer SDK will keep track of the region of
 * the frame buffer that have been invalidated. The SDK will return the
 * rectangles that have been updated through
 * VNCViewerFrameBufferUpdateEndExCallback. The SDK is able to efficiently
 * monitor the updated areas and minimize the number of rectangles that need
 * updating. Using this mode, as opposed to the default option, will therefore
 * lead to improved performance for the client application.
 *
 * \see VNCViewerSetParameter, VNCViewerGetParameter,
 * VNCParameterInvalidRegionTracking,
 * VNCViewerFrameBufferUpdateEndExCallback
 */
#define VNCViewerInvalidRegionTrackingFull "full"

#ifdef __cplusplus
}
#endif

#endif /* !defined(__VNCTYPES_H__) */
